function bound01(n, max) {
  if (isOnePointZero(n)) {
    n = `\u0031\u0030\u0030\u0025`;
  }
  var isPercent = isPercentage(n);
  n = max === 360 ? n : Math.min(max, Math.max(0, parseFloat(n)));
  if (isPercent) {
    n = parseInt(String(n * max), 10) / 100;
  }
  if (Math.abs(n - max) < 1e-6) {
    return 1;
  }
  if (max === 360) {
    n = (n < 0 ? n % max + max : n % max) / parseFloat(String(max));
  } else {
    n = n % max / parseFloat(String(max));
  }
  return n;
}
function clamp01(val) {
  return Math.min(1, Math.max(0, val));
}
function isOnePointZero(n) {
  return typeof n === `\u0073\u0074\u0072\u0069\u006e\u0067` && n.indexOf(`\u002e`) !== -1 && parseFloat(n) === 1;
}
function isPercentage(n) {
  return typeof n === `\u0073\u0074\u0072\u0069\u006e\u0067` && n.indexOf(`\u0025`) !== -1;
}
function boundAlpha(a) {
  a = parseFloat(a);
  if (isNaN(a) || a < 0 || a > 1) {
    a = 1;
  }
  return a;
}
function convertToPercentage(n) {
  if (n <= 1) {
    return ``.concat(Number(n) * 100, `\u0025`);
  }
  return n;
}
function pad2(c) {
  return c.length === 1 ? `\u0030` + c : String(c);
}
function rgbToRgb(r, g, b) {
  return {
    r: bound01(r, 255) * 255,
    g: bound01(g, 255) * 255,
    b: bound01(b, 255) * 255
  };
}
function rgbToHsl(r, g, b) {
  r = bound01(r, 255);
  g = bound01(g, 255);
  b = bound01(b, 255);
  var max = Math.max(r, g, b);
  var min = Math.min(r, g, b);
  var h = 0;
  var s = 0;
  var l = (max + min) / 2;
  if (max === min) {
    s = 0;
    h = 0;
  } else {
    var d = max - min;
    s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
    switch (max) {
      case r:
        h = (g - b) / d + (g < b ? 6 : 0);
        break;
      case g:
        h = (b - r) / d + 2;
        break;
      case b:
        h = (r - g) / d + 4;
        break;
    }
    h /= 6;
  }
  return {
    h,
    s,
    l
  };
}
function hue2rgb(p, q, t) {
  if (t < 0) {
    t += 1;
  }
  if (t > 1) {
    t -= 1;
  }
  if (t < 1 / 6) {
    return p + (q - p) * (6 * t);
  }
  if (t < 1 / 2) {
    return q;
  }
  if (t < 2 / 3) {
    return p + (q - p) * (2 / 3 - t) * 6;
  }
  return p;
}
function hslToRgb(h, s, l) {
  var r;
  var g;
  var b;
  h = bound01(h, 360);
  s = bound01(s, 100);
  l = bound01(l, 100);
  if (s === 0) {
    g = l;
    b = l;
    r = l;
  } else {
    var q = l < 0.5 ? l * (1 + s) : l + s - l * s;
    var p = 2 * l - q;
    r = hue2rgb(p, q, h + 1 / 3);
    g = hue2rgb(p, q, h);
    b = hue2rgb(p, q, h - 1 / 3);
  }
  return {
    r: r * 255,
    g: g * 255,
    b: b * 255
  };
}
function rgbToHsv(r, g, b) {
  r = bound01(r, 255);
  g = bound01(g, 255);
  b = bound01(b, 255);
  var max = Math.max(r, g, b);
  var min = Math.min(r, g, b);
  var h = 0;
  var v = max;
  var d = max - min;
  var s = max === 0 ? 0 : d / max;
  if (max === min) {
    h = 0;
  } else {
    switch (max) {
      case r:
        h = (g - b) / d + (g < b ? 6 : 0);
        break;
      case g:
        h = (b - r) / d + 2;
        break;
      case b:
        h = (r - g) / d + 4;
        break;
    }
    h /= 6;
  }
  return {
    h,
    s,
    v
  };
}
function hsvToRgb(h, s, v) {
  h = bound01(h, 360) * 6;
  s = bound01(s, 100);
  v = bound01(v, 100);
  var i = Math.floor(h);
  var f = h - i;
  var p = v * (1 - s);
  var q = v * (1 - f * s);
  var t = v * (1 - (1 - f) * s);
  var mod = i % 6;
  var r = [v, q, p, p, t, v][mod];
  var g = [t, v, v, q, p, p][mod];
  var b = [p, p, t, v, v, q][mod];
  return {
    r: r * 255,
    g: g * 255,
    b: b * 255
  };
}
function rgbToHex(r, g, b, allow3Char) {
  var hex = [pad2(Math.round(r).toString(16)), pad2(Math.round(g).toString(16)), pad2(Math.round(b).toString(16))];
  if (allow3Char && hex[0].startsWith(hex[0].charAt(1)) && hex[1].startsWith(hex[1].charAt(1)) && hex[2].startsWith(hex[2].charAt(1))) {
    return hex[0].charAt(0) + hex[1].charAt(0) + hex[2].charAt(0);
  }
  return hex.join(``);
}
function rgbaToHex(r, g, b, a, allow4Char) {
  var hex = [pad2(Math.round(r).toString(16)), pad2(Math.round(g).toString(16)), pad2(Math.round(b).toString(16)), pad2(convertDecimalToHex(a))];
  if (allow4Char && hex[0].startsWith(hex[0].charAt(1)) && hex[1].startsWith(hex[1].charAt(1)) && hex[2].startsWith(hex[2].charAt(1)) && hex[3].startsWith(hex[3].charAt(1))) {
    return hex[0].charAt(0) + hex[1].charAt(0) + hex[2].charAt(0) + hex[3].charAt(0);
  }
  return hex.join(``);
}
function convertDecimalToHex(d) {
  return Math.round(parseFloat(d) * 255).toString(16);
}
function convertHexToDecimal(h) {
  return parseIntFromHex(h) / 255;
}
function parseIntFromHex(val) {
  return parseInt(val, 16);
}
function numberInputToObject(color) {
  return {
    r: color >> 16,
    g: (color & 65280) >> 8,
    b: color & 255
  };
}
var names = {
  aliceblue: `\u0023\u0066\u0030\u0066\u0038\u0066\u0066`,
  antiquewhite: `\u0023\u0066\u0061\u0065\u0062\u0064\u0037`,
  aqua: `\u0023\u0030\u0030\u0066\u0066\u0066\u0066`,
  aquamarine: `\u0023\u0037\u0066\u0066\u0066\u0064\u0034`,
  azure: `\u0023\u0066\u0030\u0066\u0066\u0066\u0066`,
  beige: `\u0023\u0066\u0035\u0066\u0035\u0064\u0063`,
  bisque: `\u0023\u0066\u0066\u0065\u0034\u0063\u0034`,
  black: `\u0023\u0030\u0030\u0030\u0030\u0030\u0030`,
  blanchedalmond: `\u0023\u0066\u0066\u0065\u0062\u0063\u0064`,
  blue: `\u0023\u0030\u0030\u0030\u0030\u0066\u0066`,
  blueviolet: `\u0023\u0038\u0061\u0032\u0062\u0065\u0032`,
  brown: `\u0023\u0061\u0035\u0032\u0061\u0032\u0061`,
  burlywood: `\u0023\u0064\u0065\u0062\u0038\u0038\u0037`,
  cadetblue: `\u0023\u0035\u0066\u0039\u0065\u0061\u0030`,
  chartreuse: `\u0023\u0037\u0066\u0066\u0066\u0030\u0030`,
  chocolate: `\u0023\u0064\u0032\u0036\u0039\u0031\u0065`,
  coral: `\u0023\u0066\u0066\u0037\u0066\u0035\u0030`,
  cornflowerblue: `\u0023\u0036\u0034\u0039\u0035\u0065\u0064`,
  cornsilk: `\u0023\u0066\u0066\u0066\u0038\u0064\u0063`,
  crimson: `\u0023\u0064\u0063\u0031\u0034\u0033\u0063`,
  cyan: `\u0023\u0030\u0030\u0066\u0066\u0066\u0066`,
  darkblue: `\u0023\u0030\u0030\u0030\u0030\u0038\u0062`,
  darkcyan: `\u0023\u0030\u0030\u0038\u0062\u0038\u0062`,
  darkgoldenrod: `\u0023\u0062\u0038\u0038\u0036\u0030\u0062`,
  darkgray: `\u0023\u0061\u0039\u0061\u0039\u0061\u0039`,
  darkgreen: `\u0023\u0030\u0030\u0036\u0034\u0030\u0030`,
  darkgrey: `\u0023\u0061\u0039\u0061\u0039\u0061\u0039`,
  darkkhaki: `\u0023\u0062\u0064\u0062\u0037\u0036\u0062`,
  darkmagenta: `\u0023\u0038\u0062\u0030\u0030\u0038\u0062`,
  darkolivegreen: `\u0023\u0035\u0035\u0036\u0062\u0032\u0066`,
  darkorange: `\u0023\u0066\u0066\u0038\u0063\u0030\u0030`,
  darkorchid: `\u0023\u0039\u0039\u0033\u0032\u0063\u0063`,
  darkred: `\u0023\u0038\u0062\u0030\u0030\u0030\u0030`,
  darksalmon: `\u0023\u0065\u0039\u0039\u0036\u0037\u0061`,
  darkseagreen: `\u0023\u0038\u0066\u0062\u0063\u0038\u0066`,
  darkslateblue: `\u0023\u0034\u0038\u0033\u0064\u0038\u0062`,
  darkslategray: `\u0023\u0032\u0066\u0034\u0066\u0034\u0066`,
  darkslategrey: `\u0023\u0032\u0066\u0034\u0066\u0034\u0066`,
  darkturquoise: `\u0023\u0030\u0030\u0063\u0065\u0064\u0031`,
  darkviolet: `\u0023\u0039\u0034\u0030\u0030\u0064\u0033`,
  deeppink: `\u0023\u0066\u0066\u0031\u0034\u0039\u0033`,
  deepskyblue: `\u0023\u0030\u0030\u0062\u0066\u0066\u0066`,
  dimgray: `\u0023\u0036\u0039\u0036\u0039\u0036\u0039`,
  dimgrey: `\u0023\u0036\u0039\u0036\u0039\u0036\u0039`,
  dodgerblue: `\u0023\u0031\u0065\u0039\u0030\u0066\u0066`,
  firebrick: `\u0023\u0062\u0032\u0032\u0032\u0032\u0032`,
  floralwhite: `\u0023\u0066\u0066\u0066\u0061\u0066\u0030`,
  forestgreen: `\u0023\u0032\u0032\u0038\u0062\u0032\u0032`,
  fuchsia: `\u0023\u0066\u0066\u0030\u0030\u0066\u0066`,
  gainsboro: `\u0023\u0064\u0063\u0064\u0063\u0064\u0063`,
  ghostwhite: `\u0023\u0066\u0038\u0066\u0038\u0066\u0066`,
  goldenrod: `\u0023\u0064\u0061\u0061\u0035\u0032\u0030`,
  gold: `\u0023\u0066\u0066\u0064\u0037\u0030\u0030`,
  gray: `\u0023\u0038\u0030\u0038\u0030\u0038\u0030`,
  green: `\u0023\u0030\u0030\u0038\u0030\u0030\u0030`,
  greenyellow: `\u0023\u0061\u0064\u0066\u0066\u0032\u0066`,
  grey: `\u0023\u0038\u0030\u0038\u0030\u0038\u0030`,
  honeydew: `\u0023\u0066\u0030\u0066\u0066\u0066\u0030`,
  hotpink: `\u0023\u0066\u0066\u0036\u0039\u0062\u0034`,
  indianred: `\u0023\u0063\u0064\u0035\u0063\u0035\u0063`,
  indigo: `\u0023\u0034\u0062\u0030\u0030\u0038\u0032`,
  ivory: `\u0023\u0066\u0066\u0066\u0066\u0066\u0030`,
  khaki: `\u0023\u0066\u0030\u0065\u0036\u0038\u0063`,
  lavenderblush: `\u0023\u0066\u0066\u0066\u0030\u0066\u0035`,
  lavender: `\u0023\u0065\u0036\u0065\u0036\u0066\u0061`,
  lawngreen: `\u0023\u0037\u0063\u0066\u0063\u0030\u0030`,
  lemonchiffon: `\u0023\u0066\u0066\u0066\u0061\u0063\u0064`,
  lightblue: `\u0023\u0061\u0064\u0064\u0038\u0065\u0036`,
  lightcoral: `\u0023\u0066\u0030\u0038\u0030\u0038\u0030`,
  lightcyan: `\u0023\u0065\u0030\u0066\u0066\u0066\u0066`,
  lightgoldenrodyellow: `\u0023\u0066\u0061\u0066\u0061\u0064\u0032`,
  lightgray: `\u0023\u0064\u0033\u0064\u0033\u0064\u0033`,
  lightgreen: `\u0023\u0039\u0030\u0065\u0065\u0039\u0030`,
  lightgrey: `\u0023\u0064\u0033\u0064\u0033\u0064\u0033`,
  lightpink: `\u0023\u0066\u0066\u0062\u0036\u0063\u0031`,
  lightsalmon: `\u0023\u0066\u0066\u0061\u0030\u0037\u0061`,
  lightseagreen: `\u0023\u0032\u0030\u0062\u0032\u0061\u0061`,
  lightskyblue: `\u0023\u0038\u0037\u0063\u0065\u0066\u0061`,
  lightslategray: `\u0023\u0037\u0037\u0038\u0038\u0039\u0039`,
  lightslategrey: `\u0023\u0037\u0037\u0038\u0038\u0039\u0039`,
  lightsteelblue: `\u0023\u0062\u0030\u0063\u0034\u0064\u0065`,
  lightyellow: `\u0023\u0066\u0066\u0066\u0066\u0065\u0030`,
  lime: `\u0023\u0030\u0030\u0066\u0066\u0030\u0030`,
  limegreen: `\u0023\u0033\u0032\u0063\u0064\u0033\u0032`,
  linen: `\u0023\u0066\u0061\u0066\u0030\u0065\u0036`,
  magenta: `\u0023\u0066\u0066\u0030\u0030\u0066\u0066`,
  maroon: `\u0023\u0038\u0030\u0030\u0030\u0030\u0030`,
  mediumaquamarine: `\u0023\u0036\u0036\u0063\u0064\u0061\u0061`,
  mediumblue: `\u0023\u0030\u0030\u0030\u0030\u0063\u0064`,
  mediumorchid: `\u0023\u0062\u0061\u0035\u0035\u0064\u0033`,
  mediumpurple: `\u0023\u0039\u0033\u0037\u0030\u0064\u0062`,
  mediumseagreen: `\u0023\u0033\u0063\u0062\u0033\u0037\u0031`,
  mediumslateblue: `\u0023\u0037\u0062\u0036\u0038\u0065\u0065`,
  mediumspringgreen: `\u0023\u0030\u0030\u0066\u0061\u0039\u0061`,
  mediumturquoise: `\u0023\u0034\u0038\u0064\u0031\u0063\u0063`,
  mediumvioletred: `\u0023\u0063\u0037\u0031\u0035\u0038\u0035`,
  midnightblue: `\u0023\u0031\u0039\u0031\u0039\u0037\u0030`,
  mintcream: `\u0023\u0066\u0035\u0066\u0066\u0066\u0061`,
  mistyrose: `\u0023\u0066\u0066\u0065\u0034\u0065\u0031`,
  moccasin: `\u0023\u0066\u0066\u0065\u0034\u0062\u0035`,
  navajowhite: `\u0023\u0066\u0066\u0064\u0065\u0061\u0064`,
  navy: `\u0023\u0030\u0030\u0030\u0030\u0038\u0030`,
  oldlace: `\u0023\u0066\u0064\u0066\u0035\u0065\u0036`,
  olive: `\u0023\u0038\u0030\u0038\u0030\u0030\u0030`,
  olivedrab: `\u0023\u0036\u0062\u0038\u0065\u0032\u0033`,
  orange: `\u0023\u0066\u0066\u0061\u0035\u0030\u0030`,
  orangered: `\u0023\u0066\u0066\u0034\u0035\u0030\u0030`,
  orchid: `\u0023\u0064\u0061\u0037\u0030\u0064\u0036`,
  palegoldenrod: `\u0023\u0065\u0065\u0065\u0038\u0061\u0061`,
  palegreen: `\u0023\u0039\u0038\u0066\u0062\u0039\u0038`,
  paleturquoise: `\u0023\u0061\u0066\u0065\u0065\u0065\u0065`,
  palevioletred: `\u0023\u0064\u0062\u0037\u0030\u0039\u0033`,
  papayawhip: `\u0023\u0066\u0066\u0065\u0066\u0064\u0035`,
  peachpuff: `\u0023\u0066\u0066\u0064\u0061\u0062\u0039`,
  peru: `\u0023\u0063\u0064\u0038\u0035\u0033\u0066`,
  pink: `\u0023\u0066\u0066\u0063\u0030\u0063\u0062`,
  plum: `\u0023\u0064\u0064\u0061\u0030\u0064\u0064`,
  powderblue: `\u0023\u0062\u0030\u0065\u0030\u0065\u0036`,
  purple: `\u0023\u0038\u0030\u0030\u0030\u0038\u0030`,
  rebeccapurple: `\u0023\u0036\u0036\u0033\u0033\u0039\u0039`,
  red: `\u0023\u0066\u0066\u0030\u0030\u0030\u0030`,
  rosybrown: `\u0023\u0062\u0063\u0038\u0066\u0038\u0066`,
  royalblue: `\u0023\u0034\u0031\u0036\u0039\u0065\u0031`,
  saddlebrown: `\u0023\u0038\u0062\u0034\u0035\u0031\u0033`,
  salmon: `\u0023\u0066\u0061\u0038\u0030\u0037\u0032`,
  sandybrown: `\u0023\u0066\u0034\u0061\u0034\u0036\u0030`,
  seagreen: `\u0023\u0032\u0065\u0038\u0062\u0035\u0037`,
  seashell: `\u0023\u0066\u0066\u0066\u0035\u0065\u0065`,
  sienna: `\u0023\u0061\u0030\u0035\u0032\u0032\u0064`,
  silver: `\u0023\u0063\u0030\u0063\u0030\u0063\u0030`,
  skyblue: `\u0023\u0038\u0037\u0063\u0065\u0065\u0062`,
  slateblue: `\u0023\u0036\u0061\u0035\u0061\u0063\u0064`,
  slategray: `\u0023\u0037\u0030\u0038\u0030\u0039\u0030`,
  slategrey: `\u0023\u0037\u0030\u0038\u0030\u0039\u0030`,
  snow: `\u0023\u0066\u0066\u0066\u0061\u0066\u0061`,
  springgreen: `\u0023\u0030\u0030\u0066\u0066\u0037\u0066`,
  steelblue: `\u0023\u0034\u0036\u0038\u0032\u0062\u0034`,
  tan: `\u0023\u0064\u0032\u0062\u0034\u0038\u0063`,
  teal: `\u0023\u0030\u0030\u0038\u0030\u0038\u0030`,
  thistle: `\u0023\u0064\u0038\u0062\u0066\u0064\u0038`,
  tomato: `\u0023\u0066\u0066\u0036\u0033\u0034\u0037`,
  turquoise: `\u0023\u0034\u0030\u0065\u0030\u0064\u0030`,
  violet: `\u0023\u0065\u0065\u0038\u0032\u0065\u0065`,
  wheat: `\u0023\u0066\u0035\u0064\u0065\u0062\u0033`,
  white: `\u0023\u0066\u0066\u0066\u0066\u0066\u0066`,
  whitesmoke: `\u0023\u0066\u0035\u0066\u0035\u0066\u0035`,
  yellow: `\u0023\u0066\u0066\u0066\u0066\u0030\u0030`,
  yellowgreen: `\u0023\u0039\u0061\u0063\u0064\u0033\u0032`
};
function inputToRGB(color) {
  var rgb = {
    r: 0,
    g: 0,
    b: 0
  };
  var a = 1;
  var s = null;
  var v = null;
  var l = null;
  var ok = false;
  var format = false;
  if (typeof color === `\u0073\u0074\u0072\u0069\u006e\u0067`) {
    color = stringInputToObject(color);
  }
  if (typeof color === `\u006f\u0062\u006a\u0065\u0063\u0074`) {
    if (isValidCSSUnit(color.r) && isValidCSSUnit(color.g) && isValidCSSUnit(color.b)) {
      rgb = rgbToRgb(color.r, color.g, color.b);
      ok = true;
      format = String(color.r).substr(-1) === `\u0025` ? `\u0070\u0072\u0067\u0062` : `\u0072\u0067\u0062`;
    } else if (isValidCSSUnit(color.h) && isValidCSSUnit(color.s) && isValidCSSUnit(color.v)) {
      s = convertToPercentage(color.s);
      v = convertToPercentage(color.v);
      rgb = hsvToRgb(color.h, s, v);
      ok = true;
      format = `\u0068\u0073\u0076`;
    } else if (isValidCSSUnit(color.h) && isValidCSSUnit(color.s) && isValidCSSUnit(color.l)) {
      s = convertToPercentage(color.s);
      l = convertToPercentage(color.l);
      rgb = hslToRgb(color.h, s, l);
      ok = true;
      format = `\u0068\u0073\u006c`;
    }
    if (Object.prototype.hasOwnProperty.call(color, `\u0061`)) {
      a = color.a;
    }
  }
  a = boundAlpha(a);
  return {
    ok,
    format: color.format || format,
    r: Math.min(255, Math.max(rgb.r, 0)),
    g: Math.min(255, Math.max(rgb.g, 0)),
    b: Math.min(255, Math.max(rgb.b, 0)),
    a
  };
}
var CSS_INTEGER = `\u005b\u002d\u005c\u002b\u005d\u003f\u005c\u0064\u002b\u0025\u003f`;
var CSS_NUMBER = `\u005b\u002d\u005c\u002b\u005d\u003f\u005c\u0064\u002a\u005c\u002e\u005c\u0064\u002b\u0025\u003f`;
var CSS_UNIT = `\u0028\u003f\u003a`.concat(CSS_NUMBER, `\u0029\u007c\u0028\u003f\u003a`).concat(CSS_INTEGER, `\u0029`);
var PERMISSIVE_MATCH3 = `\u005b\u005c\u0073\u007c\u005c\u0028\u005d\u002b\u0028`.concat(CSS_UNIT, `\u0029\u005b\u002c\u007c\u005c\u0073\u005d\u002b\u0028`).concat(CSS_UNIT, `\u0029\u005b\u002c\u007c\u005c\u0073\u005d\u002b\u0028`).concat(CSS_UNIT, `\u0029\u005c\u0073\u002a\u005c\u0029\u003f`);
var PERMISSIVE_MATCH4 = `\u005b\u005c\u0073\u007c\u005c\u0028\u005d\u002b\u0028`.concat(CSS_UNIT, `\u0029\u005b\u002c\u007c\u005c\u0073\u005d\u002b\u0028`).concat(CSS_UNIT, `\u0029\u005b\u002c\u007c\u005c\u0073\u005d\u002b\u0028`).concat(CSS_UNIT, `\u0029\u005b\u002c\u007c\u005c\u0073\u005d\u002b\u0028`).concat(CSS_UNIT, `\u0029\u005c\u0073\u002a\u005c\u0029\u003f`);
var matchers = {
  CSS_UNIT: new RegExp(CSS_UNIT),
  rgb: new RegExp(`\u0072\u0067\u0062` + PERMISSIVE_MATCH3),
  rgba: new RegExp(`\u0072\u0067\u0062\u0061` + PERMISSIVE_MATCH4),
  hsl: new RegExp(`\u0068\u0073\u006c` + PERMISSIVE_MATCH3),
  hsla: new RegExp(`\u0068\u0073\u006c\u0061` + PERMISSIVE_MATCH4),
  hsv: new RegExp(`\u0068\u0073\u0076` + PERMISSIVE_MATCH3),
  hsva: new RegExp(`\u0068\u0073\u0076\u0061` + PERMISSIVE_MATCH4),
  hex3: /^#?([0-9a-fA-F]{1})([0-9a-fA-F]{1})([0-9a-fA-F]{1})$/,
  hex6: /^#?([0-9a-fA-F]{2})([0-9a-fA-F]{2})([0-9a-fA-F]{2})$/,
  hex4: /^#?([0-9a-fA-F]{1})([0-9a-fA-F]{1})([0-9a-fA-F]{1})([0-9a-fA-F]{1})$/,
  hex8: /^#?([0-9a-fA-F]{2})([0-9a-fA-F]{2})([0-9a-fA-F]{2})([0-9a-fA-F]{2})$/
};
function stringInputToObject(color) {
  color = color.trim().toLowerCase();
  if (color.length === 0) {
    return false;
  }
  var named = false;
  if (names[color]) {
    color = names[color];
    named = true;
  } else if (color === `\u0074\u0072\u0061\u006e\u0073\u0070\u0061\u0072\u0065\u006e\u0074`) {
    return {
      r: 0,
      g: 0,
      b: 0,
      a: 0,
      format: `\u006e\u0061\u006d\u0065`
    };
  }
  var match = matchers.rgb.exec(color);
  if (match) {
    return {
      r: match[1],
      g: match[2],
      b: match[3]
    };
  }
  match = matchers.rgba.exec(color);
  if (match) {
    return {
      r: match[1],
      g: match[2],
      b: match[3],
      a: match[4]
    };
  }
  match = matchers.hsl.exec(color);
  if (match) {
    return {
      h: match[1],
      s: match[2],
      l: match[3]
    };
  }
  match = matchers.hsla.exec(color);
  if (match) {
    return {
      h: match[1],
      s: match[2],
      l: match[3],
      a: match[4]
    };
  }
  match = matchers.hsv.exec(color);
  if (match) {
    return {
      h: match[1],
      s: match[2],
      v: match[3]
    };
  }
  match = matchers.hsva.exec(color);
  if (match) {
    return {
      h: match[1],
      s: match[2],
      v: match[3],
      a: match[4]
    };
  }
  match = matchers.hex8.exec(color);
  if (match) {
    return {
      r: parseIntFromHex(match[1]),
      g: parseIntFromHex(match[2]),
      b: parseIntFromHex(match[3]),
      a: convertHexToDecimal(match[4]),
      format: named ? `\u006e\u0061\u006d\u0065` : `\u0068\u0065\u0078\u0038`
    };
  }
  match = matchers.hex6.exec(color);
  if (match) {
    return {
      r: parseIntFromHex(match[1]),
      g: parseIntFromHex(match[2]),
      b: parseIntFromHex(match[3]),
      format: named ? `\u006e\u0061\u006d\u0065` : `\u0068\u0065\u0078`
    };
  }
  match = matchers.hex4.exec(color);
  if (match) {
    return {
      r: parseIntFromHex(match[1] + match[1]),
      g: parseIntFromHex(match[2] + match[2]),
      b: parseIntFromHex(match[3] + match[3]),
      a: convertHexToDecimal(match[4] + match[4]),
      format: named ? `\u006e\u0061\u006d\u0065` : `\u0068\u0065\u0078\u0038`
    };
  }
  match = matchers.hex3.exec(color);
  if (match) {
    return {
      r: parseIntFromHex(match[1] + match[1]),
      g: parseIntFromHex(match[2] + match[2]),
      b: parseIntFromHex(match[3] + match[3]),
      format: named ? `\u006e\u0061\u006d\u0065` : `\u0068\u0065\u0078`
    };
  }
  return false;
}
function isValidCSSUnit(color) {
  return Boolean(matchers.CSS_UNIT.exec(String(color)));
}
var TinyColor = /** @class */
function () {
  function TinyColor2(color, opts) {
    if (color === void 0) {
      color = ``;
    }
    if (opts === void 0) {
      opts = {};
    }
    var _a;
    if (color instanceof TinyColor2) {
      return color;
    }
    if (typeof color === `\u006e\u0075\u006d\u0062\u0065\u0072`) {
      color = numberInputToObject(color);
    }
    this.originalInput = color;
    var rgb = inputToRGB(color);
    this.originalInput = color;
    this.r = rgb.r;
    this.g = rgb.g;
    this.b = rgb.b;
    this.a = rgb.a;
    this.roundA = Math.round(100 * this.a) / 100;
    this.format = (_a = opts.format) !== null && _a !== void 0 ? _a : rgb.format;
    this.gradientType = opts.gradientType;
    if (this.r < 1) {
      this.r = Math.round(this.r);
    }
    if (this.g < 1) {
      this.g = Math.round(this.g);
    }
    if (this.b < 1) {
      this.b = Math.round(this.b);
    }
    this.isValid = rgb.ok;
  }
  TinyColor2.prototype.isDark = function () {
    return this.getBrightness() < 128;
  };
  TinyColor2.prototype.isLight = function () {
    return !this.isDark();
  };
  TinyColor2.prototype.getBrightness = function () {
    var rgb = this.toRgb();
    return (rgb.r * 299 + rgb.g * 587 + rgb.b * 114) / 1e3;
  };
  TinyColor2.prototype.getLuminance = function () {
    var rgb = this.toRgb();
    var R;
    var G;
    var B;
    var RsRGB = rgb.r / 255;
    var GsRGB = rgb.g / 255;
    var BsRGB = rgb.b / 255;
    if (RsRGB <= 0.03928) {
      R = RsRGB / 12.92;
    } else {
      R = Math.pow((RsRGB + 0.055) / 1.055, 2.4);
    }
    if (GsRGB <= 0.03928) {
      G = GsRGB / 12.92;
    } else {
      G = Math.pow((GsRGB + 0.055) / 1.055, 2.4);
    }
    if (BsRGB <= 0.03928) {
      B = BsRGB / 12.92;
    } else {
      B = Math.pow((BsRGB + 0.055) / 1.055, 2.4);
    }
    return 0.2126 * R + 0.7152 * G + 0.0722 * B;
  };
  TinyColor2.prototype.getAlpha = function () {
    return this.a;
  };
  TinyColor2.prototype.setAlpha = function (alpha) {
    this.a = boundAlpha(alpha);
    this.roundA = Math.round(100 * this.a) / 100;
    return this;
  };
  TinyColor2.prototype.isMonochrome = function () {
    var s = this.toHsl().s;
    return s === 0;
  };
  TinyColor2.prototype.toHsv = function () {
    var hsv = rgbToHsv(this.r, this.g, this.b);
    return {
      h: hsv.h * 360,
      s: hsv.s,
      v: hsv.v,
      a: this.a
    };
  };
  TinyColor2.prototype.toHsvString = function () {
    var hsv = rgbToHsv(this.r, this.g, this.b);
    var h = Math.round(hsv.h * 360);
    var s = Math.round(hsv.s * 100);
    var v = Math.round(hsv.v * 100);
    return this.a === 1 ? `\u0068\u0073\u0076\u0028`.concat(h, `\u002c\u0020`).concat(s, `\u0025\u002c\u0020`).concat(v, `\u0025\u0029`) : `\u0068\u0073\u0076\u0061\u0028`.concat(h, `\u002c\u0020`).concat(s, `\u0025\u002c\u0020`).concat(v, `\u0025\u002c\u0020`).concat(this.roundA, `\u0029`);
  };
  TinyColor2.prototype.toHsl = function () {
    var hsl = rgbToHsl(this.r, this.g, this.b);
    return {
      h: hsl.h * 360,
      s: hsl.s,
      l: hsl.l,
      a: this.a
    };
  };
  TinyColor2.prototype.toHslString = function () {
    var hsl = rgbToHsl(this.r, this.g, this.b);
    var h = Math.round(hsl.h * 360);
    var s = Math.round(hsl.s * 100);
    var l = Math.round(hsl.l * 100);
    return this.a === 1 ? `\u0068\u0073\u006c\u0028`.concat(h, `\u002c\u0020`).concat(s, `\u0025\u002c\u0020`).concat(l, `\u0025\u0029`) : `\u0068\u0073\u006c\u0061\u0028`.concat(h, `\u002c\u0020`).concat(s, `\u0025\u002c\u0020`).concat(l, `\u0025\u002c\u0020`).concat(this.roundA, `\u0029`);
  };
  TinyColor2.prototype.toHex = function (allow3Char) {
    if (allow3Char === void 0) {
      allow3Char = false;
    }
    return rgbToHex(this.r, this.g, this.b, allow3Char);
  };
  TinyColor2.prototype.toHexString = function (allow3Char) {
    if (allow3Char === void 0) {
      allow3Char = false;
    }
    return `\u0023` + this.toHex(allow3Char);
  };
  TinyColor2.prototype.toHex8 = function (allow4Char) {
    if (allow4Char === void 0) {
      allow4Char = false;
    }
    return rgbaToHex(this.r, this.g, this.b, this.a, allow4Char);
  };
  TinyColor2.prototype.toHex8String = function (allow4Char) {
    if (allow4Char === void 0) {
      allow4Char = false;
    }
    return `\u0023` + this.toHex8(allow4Char);
  };
  TinyColor2.prototype.toHexShortString = function (allowShortChar) {
    if (allowShortChar === void 0) {
      allowShortChar = false;
    }
    return this.a === 1 ? this.toHexString(allowShortChar) : this.toHex8String(allowShortChar);
  };
  TinyColor2.prototype.toRgb = function () {
    return {
      r: Math.round(this.r),
      g: Math.round(this.g),
      b: Math.round(this.b),
      a: this.a
    };
  };
  TinyColor2.prototype.toRgbString = function () {
    var r = Math.round(this.r);
    var g = Math.round(this.g);
    var b = Math.round(this.b);
    return this.a === 1 ? `\u0072\u0067\u0062\u0028`.concat(r, `\u002c\u0020`).concat(g, `\u002c\u0020`).concat(b, `\u0029`) : `\u0072\u0067\u0062\u0061\u0028`.concat(r, `\u002c\u0020`).concat(g, `\u002c\u0020`).concat(b, `\u002c\u0020`).concat(this.roundA, `\u0029`);
  };
  TinyColor2.prototype.toPercentageRgb = function () {
    var fmt = function (x) {
      return ``.concat(Math.round(bound01(x, 255) * 100), `\u0025`);
    };
    return {
      r: fmt(this.r),
      g: fmt(this.g),
      b: fmt(this.b),
      a: this.a
    };
  };
  TinyColor2.prototype.toPercentageRgbString = function () {
    var rnd = function (x) {
      return Math.round(bound01(x, 255) * 100);
    };
    return this.a === 1 ? `\u0072\u0067\u0062\u0028`.concat(rnd(this.r), `\u0025\u002c\u0020`).concat(rnd(this.g), `\u0025\u002c\u0020`).concat(rnd(this.b), `\u0025\u0029`) : `\u0072\u0067\u0062\u0061\u0028`.concat(rnd(this.r), `\u0025\u002c\u0020`).concat(rnd(this.g), `\u0025\u002c\u0020`).concat(rnd(this.b), `\u0025\u002c\u0020`).concat(this.roundA, `\u0029`);
  };
  TinyColor2.prototype.toName = function () {
    if (this.a === 0) {
      return `\u0074\u0072\u0061\u006e\u0073\u0070\u0061\u0072\u0065\u006e\u0074`;
    }
    if (this.a < 1) {
      return false;
    }
    var hex = `\u0023` + rgbToHex(this.r, this.g, this.b, false);
    for (var _i = 0, _a = Object.entries(names); _i < _a.length; _i++) {
      var _b = _a[_i],
        key = _b[0],
        value = _b[1];
      if (hex === value) {
        return key;
      }
    }
    return false;
  };
  TinyColor2.prototype.toString = function (format) {
    var formatSet = Boolean(format);
    format = format !== null && format !== void 0 ? format : this.format;
    var formattedString = false;
    var hasAlpha = this.a < 1 && this.a >= 0;
    var needsAlphaFormat = !formatSet && hasAlpha && (format.startsWith(`\u0068\u0065\u0078`) || format === `\u006e\u0061\u006d\u0065`);
    if (needsAlphaFormat) {
      if (format === `\u006e\u0061\u006d\u0065` && this.a === 0) {
        return this.toName();
      }
      return this.toRgbString();
    }
    if (format === `\u0072\u0067\u0062`) {
      formattedString = this.toRgbString();
    }
    if (format === `\u0070\u0072\u0067\u0062`) {
      formattedString = this.toPercentageRgbString();
    }
    if (format === `\u0068\u0065\u0078` || format === `\u0068\u0065\u0078\u0036`) {
      formattedString = this.toHexString();
    }
    if (format === `\u0068\u0065\u0078\u0033`) {
      formattedString = this.toHexString(true);
    }
    if (format === `\u0068\u0065\u0078\u0034`) {
      formattedString = this.toHex8String(true);
    }
    if (format === `\u0068\u0065\u0078\u0038`) {
      formattedString = this.toHex8String();
    }
    if (format === `\u006e\u0061\u006d\u0065`) {
      formattedString = this.toName();
    }
    if (format === `\u0068\u0073\u006c`) {
      formattedString = this.toHslString();
    }
    if (format === `\u0068\u0073\u0076`) {
      formattedString = this.toHsvString();
    }
    return formattedString || this.toHexString();
  };
  TinyColor2.prototype.toNumber = function () {
    return (Math.round(this.r) << 16) + (Math.round(this.g) << 8) + Math.round(this.b);
  };
  TinyColor2.prototype.clone = function () {
    return new TinyColor2(this.toString());
  };
  TinyColor2.prototype.lighten = function (amount) {
    if (amount === void 0) {
      amount = 10;
    }
    var hsl = this.toHsl();
    hsl.l += amount / 100;
    hsl.l = clamp01(hsl.l);
    return new TinyColor2(hsl);
  };
  TinyColor2.prototype.brighten = function (amount) {
    if (amount === void 0) {
      amount = 10;
    }
    var rgb = this.toRgb();
    rgb.r = Math.max(0, Math.min(255, rgb.r - Math.round(255 * -(amount / 100))));
    rgb.g = Math.max(0, Math.min(255, rgb.g - Math.round(255 * -(amount / 100))));
    rgb.b = Math.max(0, Math.min(255, rgb.b - Math.round(255 * -(amount / 100))));
    return new TinyColor2(rgb);
  };
  TinyColor2.prototype.darken = function (amount) {
    if (amount === void 0) {
      amount = 10;
    }
    var hsl = this.toHsl();
    hsl.l -= amount / 100;
    hsl.l = clamp01(hsl.l);
    return new TinyColor2(hsl);
  };
  TinyColor2.prototype.tint = function (amount) {
    if (amount === void 0) {
      amount = 10;
    }
    return this.mix(`\u0077\u0068\u0069\u0074\u0065`, amount);
  };
  TinyColor2.prototype.shade = function (amount) {
    if (amount === void 0) {
      amount = 10;
    }
    return this.mix(`\u0062\u006c\u0061\u0063\u006b`, amount);
  };
  TinyColor2.prototype.desaturate = function (amount) {
    if (amount === void 0) {
      amount = 10;
    }
    var hsl = this.toHsl();
    hsl.s -= amount / 100;
    hsl.s = clamp01(hsl.s);
    return new TinyColor2(hsl);
  };
  TinyColor2.prototype.saturate = function (amount) {
    if (amount === void 0) {
      amount = 10;
    }
    var hsl = this.toHsl();
    hsl.s += amount / 100;
    hsl.s = clamp01(hsl.s);
    return new TinyColor2(hsl);
  };
  TinyColor2.prototype.greyscale = function () {
    return this.desaturate(100);
  };
  TinyColor2.prototype.spin = function (amount) {
    var hsl = this.toHsl();
    var hue = (hsl.h + amount) % 360;
    hsl.h = hue < 0 ? 360 + hue : hue;
    return new TinyColor2(hsl);
  };
  TinyColor2.prototype.mix = function (color, amount) {
    if (amount === void 0) {
      amount = 50;
    }
    var rgb1 = this.toRgb();
    var rgb2 = new TinyColor2(color).toRgb();
    var p = amount / 100;
    var rgba = {
      r: (rgb2.r - rgb1.r) * p + rgb1.r,
      g: (rgb2.g - rgb1.g) * p + rgb1.g,
      b: (rgb2.b - rgb1.b) * p + rgb1.b,
      a: (rgb2.a - rgb1.a) * p + rgb1.a
    };
    return new TinyColor2(rgba);
  };
  TinyColor2.prototype.analogous = function (results, slices) {
    if (results === void 0) {
      results = 6;
    }
    if (slices === void 0) {
      slices = 30;
    }
    var hsl = this.toHsl();
    var part = 360 / slices;
    var ret = [this];
    for (hsl.h = (hsl.h - (part * results >> 1) + 720) % 360; --results;) {
      hsl.h = (hsl.h + part) % 360;
      ret.push(new TinyColor2(hsl));
    }
    return ret;
  };
  TinyColor2.prototype.complement = function () {
    var hsl = this.toHsl();
    hsl.h = (hsl.h + 180) % 360;
    return new TinyColor2(hsl);
  };
  TinyColor2.prototype.monochromatic = function (results) {
    if (results === void 0) {
      results = 6;
    }
    var hsv = this.toHsv();
    var h = hsv.h;
    var s = hsv.s;
    var v = hsv.v;
    var res = [];
    var modification = 1 / results;
    while (results--) {
      res.push(new TinyColor2({
        h,
        s,
        v
      }));
      v = (v + modification) % 1;
    }
    return res;
  };
  TinyColor2.prototype.splitcomplement = function () {
    var hsl = this.toHsl();
    var h = hsl.h;
    return [this, new TinyColor2({
      h: (h + 72) % 360,
      s: hsl.s,
      l: hsl.l
    }), new TinyColor2({
      h: (h + 216) % 360,
      s: hsl.s,
      l: hsl.l
    })];
  };
  TinyColor2.prototype.onBackground = function (background) {
    var fg = this.toRgb();
    var bg = new TinyColor2(background).toRgb();
    var alpha = fg.a + bg.a * (1 - fg.a);
    return new TinyColor2({
      r: (fg.r * fg.a + bg.r * bg.a * (1 - fg.a)) / alpha,
      g: (fg.g * fg.a + bg.g * bg.a * (1 - fg.a)) / alpha,
      b: (fg.b * fg.a + bg.b * bg.a * (1 - fg.a)) / alpha,
      a: alpha
    });
  };
  TinyColor2.prototype.triad = function () {
    return this.polyad(3);
  };
  TinyColor2.prototype.tetrad = function () {
    return this.polyad(4);
  };
  TinyColor2.prototype.polyad = function (n) {
    var hsl = this.toHsl();
    var h = hsl.h;
    var result = [this];
    var increment = 360 / n;
    for (var i = 1; i < n; i++) {
      result.push(new TinyColor2({
        h: (h + i * increment) % 360,
        s: hsl.s,
        l: hsl.l
      }));
    }
    return result;
  };
  TinyColor2.prototype.equals = function (color) {
    return this.toRgbString() === new TinyColor2(color).toRgbString();
  };
  return TinyColor2;
}();
export { TinyColor as T };